import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion } from 'framer-motion'
import { Calendar, Code, CheckCircle, ArrowRight, Layers, CheckCircle2, Users, Clock, TrendingUp, Star } from 'lucide-react'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedProgram, setSelectedProgram] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const programs = [
    { 
      id: 'blender', 
      name: 'Blender', 
      icon: Code,
      color: '#1A1F2E',
      description: 'Бесплатное и мощное ПО для 3D-моделирования',
      duration: '8 недель',
      level: 'Начинающий',
      students: '2500+'
    },
    { 
      id: 'maya', 
      name: 'Maya', 
      icon: Layers,
      color: '#1A1F2E',
      description: 'Профессиональный стандарт индустрии',
      duration: '10 недель',
      level: 'Продвинутый',
      students: '1800+'
    },
    { 
      id: '3dsmax', 
      name: '3ds Max', 
      icon: TrendingUp,
      color: '#1A1F2E',
      description: 'Мощный инструмент для архитектуры и визуализации',
      duration: '12 недель',
      level: 'Профессионал',
      students: '3200+'
    },
    { 
      id: 'zbrush', 
      name: 'ZBrush', 
      icon: Star,
      color: '#1A1F2E',
      description: 'Цифровая скульптура и детализация',
      duration: '9 недель',
      level: 'Продвинутый',
      students: '2100+'
    }
  ]

  const instructors = [
    { name: 'Алексей Волков', program: 'Blender', experience: '12 лет', rating: 5, students: '2500+' },
    { name: 'Мария Соколова', program: 'Maya', experience: '15 лет', rating: 5, students: '1800+' },
    { name: 'Дмитрий Новиков', program: '3ds Max', experience: '10 лет', rating: 5, students: '3200+' },
    { name: 'Елена Петрова', program: 'ZBrush', experience: '8 лет', rating: 5, students: '2100+' }
  ]

  const benefits = [
    { icon: CheckCircle2, title: 'Структурированная программа', text: 'Пошаговое обучение от основ до продвинутых техник' },
    { icon: Layers, title: 'Практические проекты', text: 'Реальные задачи и портфолио для вашей карьеры' },
    { icon: Users, title: 'Обратная связь', text: 'Персональные консультации с опытными преподавателями' },
    { icon: Star, title: 'Сертификаты', text: 'Официальные документы о прохождении курсов' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedProgram && startDate) {
      const savedProjects = localStorage.getItem('model3dProjects')
      const projects = savedProjects ? JSON.parse(savedProjects) : []
      
      const programData = programs.find(p => p.id === selectedProgram)
      const instructor = instructors.find(i => i.program === programData?.name)
      
      const newProject = {
        id: Date.now(),
        course: programData?.name || 'Курс',
        instructor: instructor?.name || 'Преподаватель',
        submissionDate: startDate,
        rating: 0,
        progress: 0
      }

      projects.push(newProject)
      localStorage.setItem('model3dProjects', JSON.stringify(projects))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedProgram('')
        setStartDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll">
      <div className="container">
        <motion.section 
          className="enroll-hero-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="enroll-hero-clean-content">
            <h1 className="enroll-clean-title">Записаться на курс</h1>
            <p className="enroll-clean-description">
              Выберите программу и дату начала обучения
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="enroll-form-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <div className="enroll-form-clean-wrapper">
            <form className="enroll-form-clean-inner" onSubmit={handleSubmit}>
              <div className="form-clean-section">
                <div className="form-clean-label">
                  <Code size={28} strokeWidth={2} className="label-clean-icon" />
                  <span className="label-clean-text">Программа обучения</span>
                </div>
                <div className="programs-clean-grid">
                  {programs.map((program, index) => {
                    const IconComponent = program.icon
                    return (
                      <motion.button
                        key={program.id}
                        type="button"
                        className={`program-clean-card ${selectedProgram === program.id ? 'selected' : ''}`}
                        onClick={() => setSelectedProgram(program.id)}
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: 0.3 + index * 0.1, duration: 0.5 }}
                        whileHover={{ y: -4, transition: { duration: 0.2 } }}
                      >
                        <div className="program-clean-icon" style={{ 
                          background: selectedProgram === program.id ? 'rgba(26, 31, 46, 0.1)' : 'rgba(26, 31, 46, 0.05)',
                          borderColor: selectedProgram === program.id ? program.color : 'rgba(26, 31, 46, 0.2)'
                        }}>
                          <IconComponent size={32} strokeWidth={2} color={selectedProgram === program.id ? program.color : '#1A1F2E'} />
                        </div>
                        <div className="program-clean-content">
                          <h3 className="program-clean-name">{program.name}</h3>
                          <p className="program-clean-description">{program.description}</p>
                          <div className="program-clean-meta">
                            <div className="program-clean-meta-item">
                              <Clock size={14} strokeWidth={2} />
                              <span>{program.duration}</span>
                            </div>
                            <div className="program-clean-meta-item">
                              <TrendingUp size={14} strokeWidth={2} />
                              <span>{program.level}</span>
                            </div>
                            <div className="program-clean-meta-item">
                              <Users size={14} strokeWidth={2} />
                              <span>{program.students}</span>
                            </div>
                          </div>
                        </div>
                        {selectedProgram === program.id && (
                          <motion.div 
                            className="program-clean-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            transition={{ type: "spring", stiffness: 200, damping: 15 }}
                          >
                            <CheckCircle size={24} strokeWidth={2} color={program.color} />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              {selectedProgram && (
                <motion.div 
                  className="form-clean-section date-clean-section"
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  transition={{ duration: 0.5 }}
                >
                  <div className="date-clean-label">
                    <Calendar size={28} strokeWidth={2} className="label-clean-icon" />
                    <span className="date-clean-label-text">Дата начала обучения</span>
                  </div>
                  <input
                    type="date"
                    id="startDate"
                    className="form-clean-input"
                    value={startDate}
                    onChange={(e) => setStartDate(e.target.value)}
                    min={minDate}
                    max={maxDate}
                    required
                  />
                </motion.div>
              )}

              <motion.button 
                type="submit" 
                className="submit-clean-button" 
                disabled={!selectedProgram || !startDate}
                whileHover={{ scale: 1.02, y: -2 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на курс</span>
                <ArrowRight size={24} strokeWidth={2} className="btn-clean-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-clean-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.4 }}
                >
                  <CheckCircle size={32} strokeWidth={2} className="success-clean-icon" />
                  <span>Вы успешно записаны на курс!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="instructors-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <div className="instructors-clean-header">
            <h2 className="section-clean-title">Наши преподаватели</h2>
          </div>
          <div className="instructors-clean-grid">
            {instructors.map((instructor, index) => (
              <motion.div
                key={instructor.name}
                className="instructor-clean-card"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.4 + index * 0.1, duration: 0.5 }}
                whileHover={{ y: -4, transition: { duration: 0.2 } }}
              >
                <div className="instructor-clean-avatar">
                  <Users size={36} strokeWidth={2} color="#1A1F2E" />
                </div>
                <h3 className="instructor-clean-name">{instructor.name}</h3>
                <div className="instructor-clean-program">{instructor.program}</div>
                <div className="instructor-clean-details">
                  <div className="instructor-clean-detail">
                    <Clock size={14} strokeWidth={2} />
                    <span>{instructor.experience}</span>
                  </div>
                  <div className="instructor-clean-detail">
                    <Star size={14} strokeWidth={2} fill="#1A1F2E" color="#1A1F2E" />
                    <span>{instructor.rating}.0</span>
                  </div>
                  <div className="instructor-clean-detail">
                    <Users size={14} strokeWidth={2} />
                    <span>{instructor.students}</span>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="benefits-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.4 }}
        >
          <div className="benefits-clean-header">
            <h2 className="section-clean-title">Преимущества обучения</h2>
          </div>
          <div className="benefits-clean-grid">
            {benefits.map((benefit, index) => {
              const IconComponent = benefit.icon
              return (
                <motion.div
                  key={benefit.title}
                  className="benefit-clean-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.5 + index * 0.1, duration: 0.5 }}
                  whileHover={{ y: -4, transition: { duration: 0.2 } }}
                >
                  <div className="benefit-clean-icon">
                    <IconComponent size={32} strokeWidth={2} color="#1A1F2E" />
                  </div>
                  <h3 className="benefit-clean-title">{benefit.title}</h3>
                  <p className="benefit-clean-text">{benefit.text}</p>
                  <div className="benefit-clean-accent"></div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default Enroll

